// PCI_LIB.H

#ifndef _PCI_LIB_H
#define _PCI_LIB_H


#ifdef __cplusplus
extern "C" {
#endif


#include <windows.h> // for BOOL, DWORD etc

/*
 This file encompasses the PCI driver interface and is used by several
 applications: both C and C++, so it was written in C.
 Users of the C++ library will not need to call any of its functions
 directly.  Nor should they need to be familiar with its contents.

 The functions described in this file are provided with the C++ library
 in a DLL (PCI_LIB.DLL).  PCI_LIB.DLL makes use of a system driver file:
 either WINDRVR.VXD (for Windows 95)
 or     WINDRVR.SYS (for Windows 98/ME/NT/2000/XP)
 PCI_LIB.DLL and both WINDRVR files are provided with the C++ library
 as well as the PCI loop controller installation disk.

 The following interface files are provided with the C++ library for easy
 coupling to PCI_LIB.DLL.
 For C++ : PCI_MEM.H/CPP
 For C   : PCI_MEMC.H/C

 As an added option, the C++ library provides the files WIN32MEM.H/CPP
 which takes care of all the coupling to both PCI and ISA loop controllers
 for any Windows operating system.  WIN32MEM can be used in place of the
 PCI_MEM, 9X_MEM and NT_MEM files.

More complete information on programming an interface to the
Group3 PCI loop controller is provided with the C++ library README.DOC.


Version numbers as returned by pci_get_lib_version():
Version 1.00
  RELEASED Jan-01

Version 1.01
  Added typedefs:  PCI_LC_Diag_Int_Handler_LC
                   PCI_LC_Norm_Int_Handler_LC
  Added functions: pci_enable_int_diag_lc()
                   pci_enable_int_norm_lc()
  These allow assigning an interrupt sub routine (ISR) that will be passed
  the number of the loop controller that generated the interrupt.  This is
  useful if an ISR is used to service multiple loop controllers.
  RELEASED: 08-Nov-01

Version 1.02
  Upgraded to WINDRVR version 6.00
  RELEASED 29-Apr-03

Version 1.03
  Upgraded to WINDRVR version 6.01+ (+ because transient version from Jungo)
  RELEASED 14-May-03
*/


#define DLLEXPORT __declspec(dllexport)


// general interrupt stuff:
// Note: using interrupts with the PCI loop controllers is optional.

// structure passed to application interrupt handler (diagnostic handler only)
typedef struct PCI_Int_Result
{
    DWORD counter; // number of interrupts received
    DWORD lost;    // number of interrupts not yet dealt with
    BOOL  stopped; // was interrupt disabled during wait
} PCI_Int_Result;

// two interrupt functions which are not passed the LC number
// diagnostic interrupt handler: returns information about interrupt handling performance
typedef void ( WINAPI *PCI_LC_Diag_Int_Handler    )( PCI_Int_Result *int_result );
// normal interrupt handler: less overhead than the diagnostic version
typedef void ( WINAPI *PCI_LC_Norm_Int_Handler    )( void );

// two interrupt functions which are     passed the LC number
// diagnostic interrupt handler: returns information about interrupt handling performance
typedef void ( WINAPI *PCI_LC_Diag_Int_Handler_LC )( int lc_no, PCI_Int_Result *int_result );
// normal interrupt handler: less overhead than the diagnostic version
typedef void ( WINAPI *PCI_LC_Norm_Int_Handler_LC )( int lc_no );


enum { MAX_PCI_LCS     = 16 }; // 0..15
enum { MAX_PCI_LC_ADDR = 15 }; // 0..15
/*
 MAX_PCI_LCS: A maximum of 16 loop controllers is possible through selection
 with rotary switches.  An LC3 has three loop controllers and each one comes
 off the maximum total of 16.

 The maximum number of LC cards that can be used is also 16.
 The maximum number of loop controllers can be met with a combination of 
 LC1's and LC3's.  eg 16 LC1's or Five LC3's and one LC1.
 If more loop controllers are present, then there will be overlaps.
 Likewise the software will recognise up to 16 LC3's, however there will be
 many overlaps.

 Overlaps are permissible, however for two LC cards with a loop controller at
 the same address, the loop controller whose LC card has the lower switch
 setting will dominate.  Also, for for two LC cards with a loop controller at
 the same address and with the same switch setting, the loop controller whose
 LC card is discovered first by the driver will dominate.  For this second
 case, predicting the dominant loop controller may not be possible.

 Generally, AVOID overlaps !!!

 If the rotary switch on an LC3 is set to 14, then its loop controllers will
 be at addresses 14,15,0.  Likewise if the switch is set to 15, the loop
 controllers will be at 15,0,1.
*/


DLLEXPORT float pci_get_lib_version( void ); // used for feature control

DLLEXPORT BOOL  pci_open_driver( void );  // must call (at start) : returns TRUE if got driver
DLLEXPORT void  pci_close_driver( void ); // must call (when finished)
DLLEXPORT BOOL  pci_register_lc( int lc_no, BOOL allow_ints ); // must call
DLLEXPORT char *pci_get_dp_access_addr( int lc_no ); // must call
// returns a pointer for direct memory access
// : use this value for the LoopController constructor in the C++ library
// : will return NULL if loop controller not mapped

// interrupt usage functions:
// call one of the pci_enable_int_...() functions, and only one !!!!
DLLEXPORT BOOL  pci_int_is_enabled(     int lc_no );
DLLEXPORT BOOL  pci_enable_int_diag(    int lc_no, PCI_LC_Diag_Int_Handler    user_int_handler, BOOL end_chain );
DLLEXPORT BOOL  pci_enable_int_norm(    int lc_no, PCI_LC_Norm_Int_Handler    user_int_handler, BOOL end_chain );
DLLEXPORT BOOL  pci_enable_int_diag_lc( int lc_no, PCI_LC_Diag_Int_Handler_LC user_int_handler, BOOL end_chain );
DLLEXPORT BOOL  pci_enable_int_norm_lc( int lc_no, PCI_LC_Norm_Int_Handler_LC user_int_handler, BOOL end_chain );
DLLEXPORT void  pci_disable_int(        int lc_no );

// diagnostic functions: can be called after pci_open_driver()
DLLEXPORT BOOL  pci_driver_avail(      void );
DLLEXPORT BOOL  pci_any_lc_overlaps(   void ); // for all cards/loop controllers
DLLEXPORT int   pci_no_of_LCs_found(   void ); // loop controllers, not LC cards
DLLEXPORT int   pci_no_of_cards_found( void ); // LC cards
DLLEXPORT BOOL  pci_lc_present(   int lc_no );
DLLEXPORT int   pci_lc_switch_no( int lc_no );
DLLEXPORT int   pci_lc_loop_no(   int lc_no );


#ifdef __cplusplus
}
#endif

#endif
